package com.teamviewer.commonresourcelib.network

import android.app.AlarmManager
import android.app.Notification
import android.app.PendingIntent
import android.app.Service
import android.content.Intent
import android.os.Build
import android.os.IBinder
import android.os.Parcelable
import androidx.annotation.MainThread
import com.teamviewer.commonresourcelib.BuildConfig
import com.teamviewer.libs.assertion.TVAssert.assertTrue
import com.teamviewer.libs.logging.Logging
import com.teamviewer.teamviewerlib.event.EventHub.Companion.instance
import com.teamviewer.teamviewerlib.gui.TVNotificationManager
import com.teamviewer.teamviewerlib.network.Network
import com.teamviewer.teamviewerlib.network.Network.startWatchdog
import com.teamviewer.teamviewerlib.network.Network.stopWatchdog
import com.teamviewer.teamviewerlib.swig.tvshared.EventType

abstract class AbstractNetworkService : Service() {
    private var isNetworkStarted = false
    private var heartbeatIntent: PendingIntent? = null
    private var notification: Notification? = null

    override fun onCreate() {
        super.onCreate()
        if (BuildConfig.DEBUG) Logging.LogDebug(TAG, "Create")
    }

    override fun onStartCommand(
        intent: Intent?,
        flags: Int,
        startId: Int,
    ): Int {
        super.onStartCommand(intent, flags, startId)
        val runInForeground =
            if (intent != null) {
                getIntentFromParcelable(intent)
                intent.getBooleanExtra(EXTRA_FOREGROUND, false)
            } else {
                notification = null
                false
            }
        if (BuildConfig.DEBUG) {
            if (flags != 0) {
                Logging.LogDebug(TAG, "Re-created")
            }
            if (intent?.getBooleanExtra(EXTRA_HEARTBEAT, false) == true) {
                Logging.LogDebug(TAG, "Received heartbeat")
            }
            if (runInForeground) {
                assertTrue(notification != null) { "Foreground services require a notification!" }
                Logging.LogDebug(TAG, "Received foreground")
            } else {
                Logging.LogDebug(TAG, "Received start command")
            }
        }

        try {
            if (runInForeground) {
                startForeground(TVNotificationManager.NOTIFICATION_ID_NETWORK_SERVICE, notification)
            } else {
                stopForegroundService()
            }
        } catch (e: Exception) {
            Logging.LogError(TAG, "Starting Foreground service failed with exception: ${e.message}")
        }

        Logging.Log(TAG, "Starting network.")
        start()
        return if (useHeartbeat()) {
            Logging.Log(TAG, "Using heartbeat")
            START_STICKY
        } else {
            Logging.LogWarning(TAG, "Heartbeat not used")
            START_NOT_STICKY
        }
    }

    private fun getIntentFromParcelable(intent: Intent) {
        notification =
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
                intent.getParcelableExtra(
                    EXTRA_NOTIFICATION,
                    Notification::class.java,
                )
            } else {
                @Suppress("deprecation")
                intent.getParcelableExtra<Parcelable>(EXTRA_NOTIFICATION) as Notification?
            }
    }

    @Suppress("deprecation")
    private fun stopForegroundService() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            stopForeground(STOP_FOREGROUND_REMOVE)
        } else {
            stopForeground(true)
        }
    }

    override fun onBind(intent: Intent): IBinder? {
        if (BuildConfig.DEBUG) Logging.LogError(TAG, "Binding not allowed")
        return null
    }

    override fun onDestroy() {
        super.onDestroy()
        if (BuildConfig.DEBUG) Logging.LogDebug(TAG, "Destroy")
        stop()
        stopForegroundService()
        Logging.Log(TAG, "Service was terminated successfully.")
    }

    override fun onTaskRemoved(rootIntent: Intent) {
        super.onTaskRemoved(rootIntent)
        if (BuildConfig.DEBUG) Logging.LogDebug(TAG, "Task removed $rootIntent")
        Logging.Log(TAG, "Task removed, scheduling network restart.")
        startHeartbeat()
        instance.triggerEvent(EventType.EVENT_APP_TASK_REMOVED)
    }

    private fun start() {
        startNetwork()
        startHeartbeat()
    }

    private fun stop() {
        stopHeartbeat()
        stopNetwork()
    }

    private fun startNetwork() {
        if (!isNetworkStarted) {
            Logging.Log(TAG, "Start network.")
            Network.startNetwork()
            startWatchdog()
            isNetworkStarted = true
        }
    }

    private fun startHeartbeat() {
        if (useHeartbeat()) {
            // Note: heartbeat will be always triggered for at least
            // 		 the first time because	we do not know
            // 		 on application startup so early on
            // 		 if the push notification service will work
            // 		 after successful initialisation
            Logging.Log(TAG, "Start heartbeat.")
            createAndStorePendingIntentIfNeeded()
            scheduleIntent()
        }
    }

    private fun stopHeartbeat() {
        if (heartbeatIntent != null) {
            Logging.Log(TAG, "Stop heartbeat.")
            val alarmManager = getSystemService(ALARM_SERVICE) as AlarmManager
            alarmManager.cancel(heartbeatIntent!!)
            heartbeatIntent = null
        }
    }

    private fun stopNetwork() {
        if (isNetworkStarted) {
            isNetworkStarted = false
            Logging.Log(TAG, "Stop network.")
            stopWatchdog()
            Network.stopNetwork()
        } else {
            Logging.LogWarning(TAG, "stopNetwork - already stopped")
        }
    }

    protected abstract fun useHeartbeat(): Boolean

    protected abstract fun getHeartbeatInterval(): Int

    @MainThread
    private fun createAndStorePendingIntentIfNeeded() {
        if (heartbeatIntent == null) {
            val intent = Intent(this, javaClass)
            intent.putExtra(EXTRA_HEARTBEAT, true)
            if (notification != null) {
                intent.putExtra(EXTRA_FOREGROUND, true)
                intent.putExtra(EXTRA_NOTIFICATION, notification)
            }
            val intentFlags: Int =
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
                    PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
                } else {
                    PendingIntent.FLAG_UPDATE_CURRENT
                }
            heartbeatIntent = PendingIntent.getService(this, 0, intent, intentFlags)
        }
    }

    @MainThread
    private fun scheduleIntent() {
        if (useHeartbeat()) {
            val alarmManager = getSystemService(ALARM_SERVICE) as AlarmManager
            alarmManager.setWindow(
                AlarmManager.RTC_WAKEUP,
                System.currentTimeMillis() + getHeartbeatInterval() - 1000,
                1000,
                heartbeatIntent!!,
            )
        }
    }

    companion object {
        private const val TAG = "NetworkService"
        const val EXTRA_FOREGROUND = "com.teamviewer.teamviewer.extra.NETWORK_FOREGROUND"
        const val EXTRA_NOTIFICATION = "com.teamviewer.teamviewer.extra.NETWORK_NOTIFICATION"
        private const val EXTRA_HEARTBEAT = "com.teamviewer.teamviewer.extra.NETWORK_HEARTBEAT"
        const val NETWORK_STANDBY_TIMEOUT_TIME = 600000 // 10 minutes in ms
        const val GRACEFUL_TERMINATION_DELAY = 1000
    }
}
